<?php
// Simple PHP endpoint to receive 20 tokens and write a JSON per server with date key
// Place this file on your web server (e.g., /var/www/html/monitor/tokens.php)

declare(strict_types=1);

header('Content-Type: application/json');

// CONFIG
$BASE_DIR = __DIR__ . '/data'; // Change as needed
$REQUIRE_AUTH = false;         // Set to true to enforce bearer token
$API_TOKEN =  'kaleUinMalang2024Secret';

function fail(int $code, string $msg) {
    http_response_code($code);
    echo json_encode(['ok' => false, 'error' => $msg], JSON_UNESCAPED_SLASHES);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    fail(405, 'Method Not Allowed');
}

if ($REQUIRE_AUTH) {
    $hdr = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    if (!preg_match('/^Bearer\s+(.*)$/i', $hdr, $m)) fail(401, 'Missing bearer token');
    if (!$API_TOKEN || trim($m[1]) !== $API_TOKEN) fail(403, 'Invalid token');
}

$raw = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) fail(400, 'Invalid JSON');

$serverId = $data['serverId'] ?? '';
$date = $data['date'] ?? '';
$tokens = $data['tokens'] ?? null; // array of { idx, value }
$ts = $data['ts'] ?? null;

if (!$serverId || !is_string($serverId)) fail(400, 'serverId required');
if (!$date || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) fail(400, 'date required YYYY-MM-DD');
if (!is_array($tokens)) fail(400, 'tokens array required');

// Normalize
$byIdx = [];
foreach ($tokens as $t) {
    if (!isset($t['idx'])) continue;
    $i = (int)$t['idx'];
    if ($i < 1 || $i > 20) continue;
    $val = isset($t['value']) && is_string($t['value']) ? $t['value'] : '';
    $byIdx[$i] = $val;
}
for ($i = 1; $i <= 20; $i++) {
    if (!isset($byIdx[$i])) $byIdx[$i] = '';
}

// Prepare paths
$dir = $BASE_DIR . '/tokens';
if (!is_dir($dir) && !@mkdir($dir, 0775, true) && !is_dir($dir)) {
    fail(500, 'Cannot create data dir');
}
$safe = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $serverId);
$file = $dir . '/' . $safe . '.json';

// Load existing file (per server)
$current = [
    'serverId' => $serverId,
    'dates' => [] // { 'YYYY-MM-DD': { updatedAt, entries: [ {idx,value}, ... ] } }
];
if (is_file($file)) {
    $raw2 = @file_get_contents($file);
    $dec = @json_decode($raw2, true);
    if (is_array($dec)) $current = $dec;
}

$current['serverId'] = $serverId; // ensure correct

// Upsert for date
$entries = [];
for ($i = 1; $i <= 20; $i++) {
    $entries[] = ['idx' => $i, 'value' => $byIdx[$i]];
}
$current['dates'][$date] = [
    'updatedAt' => (new DateTimeImmutable('now'))->format(DATE_ATOM),
    'tsClient' => $ts,
    'entries' => $entries
];

// Atomic write
$json = json_encode($current, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
if ($json === false) fail(500, 'JSON encode failed');
$tmp = $file . '.tmp';
if (@file_put_contents($tmp, $json, LOCK_EX) === false) fail(500, 'Write tmp failed');
if (!@rename($tmp, $file)) fail(500, 'Rename failed');

echo json_encode(['ok' => true, 'file' => basename($file), 'date' => $date], JSON_UNESCAPED_SLASHES);
