<?php
/*
 * Tmux Workers Endpoint
 * - Receives POST JSON with tmux session status
 * - Writes to data/tmux/<serverId>.json
 * - Atomic file write with lock
 */

header('Content-Type: application/json');

// Config
$REQUIRE_AUTH = true;
$API_TOKEN = 'kaleUinMalang2024Secret'; // Ganti sesuai monitor-config.json
$DATA_DIR = __DIR__ . '/data/tmux';

// Create data directory if not exists
if (!is_dir($DATA_DIR)) {
    mkdir($DATA_DIR, 0775, true);
}

// Authentication
if ($REQUIRE_AUTH) {
    $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    if (!$authHeader || !preg_match('/Bearer\s+(\S+)/', $authHeader, $matches)) {
        http_response_code(401);
        echo json_encode(['error' => 'Missing authorization header']);
        exit;
    }
    if ($matches[1] !== $API_TOKEN) {
        http_response_code(403);
        echo json_encode(['error' => 'Invalid token']);
        exit;
    }
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed. Use POST.']);
    exit;
}

// Parse JSON body
$rawBody = file_get_contents('php://input');
$data = json_decode($rawBody, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON']);
    exit;
}

// Validate required fields
$serverId = $data['serverId'] ?? null;
if (!$serverId || !preg_match('/^[a-zA-Z0-9_-]+$/', $serverId)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid or missing serverId']);
    exit;
}

// Build sessions array from running/missing
$sessions = [];
if (isset($data['running']) && isset($data['missing'])) {
    $prefix = 'worker_';
    foreach ($data['running'] as $idx) {
        $sessions[] = ['name' => $prefix . $idx, 'status' => 'online'];
    }
    foreach ($data['missing'] as $idx) {
        $sessions[] = ['name' => $prefix . $idx, 'status' => 'offline'];
    }
    // Sort by name
    usort($sessions, function($a, $b) {
        return strnatcmp($a['name'], $b['name']);
    });
}

// Prepare output
$output = [
    'serverId' => $serverId,
    'date' => date('Y-m-d'),
    'tsClient' => $data['ts'] ?? date('c'),
    'tsServer' => date('c'),
    'runningCount' => $data['runningCount'] ?? 0,
    'missingCount' => $data['missingCount'] ?? 0,
    'total' => $data['range'][1] ?? 250,
    'sessions' => $sessions
];

// Atomic write
$targetFile = $DATA_DIR . '/' . $serverId . '.json';
$tmpFile = $targetFile . '.tmp.' . getmypid();

$encoded = json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
if (file_put_contents($tmpFile, $encoded) === false) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to write temp file']);
    exit;
}

if (!rename($tmpFile, $targetFile)) {
    @unlink($tmpFile);
    http_response_code(500);
    echo json_encode(['error' => 'Failed to rename file']);
    exit;
}

// Success
http_response_code(200);
echo json_encode([
    'ok' => true,
    'file' => $serverId . '.json',
    'online' => $output['runningCount'],
    'offline' => $output['missingCount']
]);
