<?php
// Simple PHP endpoint to receive LaunchTube credits and write a JSON file per server
// Place this file on your web server (e.g., /var/www/html/monitor/credits.php)
// Configure writeable data directory; script will create subfolders.

declare(strict_types=1);

header('Content-Type: application/json');

// CONFIG
$BASE_DIR = __DIR__ . '/data'; // Change to an absolute path outside web root if preferred
$REQUIRE_AUTH = false;         // Set to true to enforce bearer token
$API_TOKEN = 'kaleUinMalang2024Secret'; // or hardcode e.g., 'mysecrettoken'

function fail(int $code, string $msg) {
    http_response_code($code);
    echo json_encode(['ok' => false, 'error' => $msg], JSON_UNESCAPED_SLASHES);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    fail(405, 'Method Not Allowed');
}

if ($REQUIRE_AUTH) {
    $hdr = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    if (!preg_match('/^Bearer\s+(.*)$/i', $hdr, $m)) fail(401, 'Missing bearer token');
    if (!$API_TOKEN || trim($m[1]) !== $API_TOKEN) fail(403, 'Invalid token');
}

$raw = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) fail(400, 'Invalid JSON');

$serverId = $data['serverId'] ?? '';
$creditsRaw = $data['creditsRaw'] ?? null;
$creditsXLM = $data['creditsXLM'] ?? null;
$ts = $data['ts'] ?? null;
if (!$serverId || !is_string($serverId)) fail(400, 'serverId required');
if (!is_numeric($creditsRaw) || !is_numeric($creditsXLM)) fail(400, 'creditsRaw and creditsXLM required');

$now = (new DateTimeImmutable('now'));
$payload = [
    'serverId'   => $serverId,
    'date'       => $now->format('Y-m-d'),
    'tsClient'   => $ts,
    'tsServer'   => $now->format(DATE_ATOM),
    'creditsRaw' => (int)$creditsRaw,
    'creditsXLM' => (float)$creditsXLM,
];

// Prepare paths
$dir = $BASE_DIR . '/credits';
if (!is_dir($dir) && !@mkdir($dir, 0775, true) && !is_dir($dir)) {
    fail(500, 'Cannot create data dir');
}
// sanitize serverId for filename
$safe = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $serverId);
$file = $dir . '/' . $safe . '.json';
$tmp  = $file . '.tmp';

$json = json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
if ($json === false) fail(500, 'JSON encode failed');

// Atomic write
if (@file_put_contents($tmp, $json, LOCK_EX) === false) fail(500, 'Write tmp failed');
if (!@rename($tmp, $file)) fail(500, 'Rename failed');

echo json_encode(['ok' => true, 'file' => basename($file)], JSON_UNESCAPED_SLASHES);
